from dataclasses import asdict, dataclass
from typing import Any, DefaultDict, Dict, List, Optional, Tuple

from pyrallis import field


@dataclass
class OfflineModelTrainConfig:
    # wandb params
    project: str = "OSRL-baselines"
    group: str = None
    name: Optional[str] = None
    prefix: Optional[str] = "environment_model"
    suffix: Optional[str] = ""
    logdir: Optional[str] = "logs"
    verbose: bool = True
    # dataset params
    outliers_percent: float = None
    noise_scale: float = None
    inpaint_ranges: Tuple[Tuple[float, float], ...] = None
    epsilon: float = None
    density: float = 1.0
    # model params
    safe_only: bool = False
    simple_scaler: bool = False
    dynamic_hidden_dims: List[int] = field(default=[256,256,256,256], is_mutable=True)
    cost_model_hidden_dims: List[int] = field(default=[1024,1024,1024,1024], is_mutable=True)
    num_ensemble: int = 7
    num_elites: int = 5
    dynamic_weight_decays: List[float] = field(default=[2.5e-5, 5e-5, 7.5e-5, 7.5e-5, 1e-4], is_mutable=True)
    with_reward: bool = True
    with_cost: bool = False
    use_delta_obs: bool = True
    penalty_coef: float = 2.5
    cost_coef: float = 0.5
    max_epochs_since_update: int = 5
    uncertainty_mode: str = "aleatoric"
    # training params
    task: str = "OfflineCarCircle-v0"
    # task: str = "OfflineAntCircleGymnasium-v1"
    dataset: str = None
    learning_rate: float = 1e-3
    use_scheduler: bool = False
    decay_step: int = 1000
    decay_rate: int = 0.95
    batch_size: int = 4096
    reward_scale: float = 1.0
    cost_scale: float = 1
    num_workers: int = 8
    # evaluation params
    cost_limit: int = 10
    # general params
    seed: int = 0
    device: str = "cpu"
    threads: int = 6
    # augmentation param
    deg: int = 4
    pf_sample: bool = False
    beta: float = 1.0
    augment_percent: float = 0.2
    cost_reverse: bool = False
    # maximum absolute value of reward for the augmented trajs
    max_reward: float = 600.0
    # minimum reward above the PF curve
    min_reward: float = 1.0
    # the max drecrease of ret between the associated traj
    # w.r.t the nearest pf traj
    max_rew_decrease: float = 100.0
    # pf only mode param
    pf_only: bool = False
    rmin: float = 300
    cost_bins: int = 60
    npb: int = 5
    cost_sample: bool = False
    linear: bool = True  # linear or inverse
    start_sampling: bool = False
    prob: float = 0.2
    stochastic: bool = True
    init_temperature: float = 0.1
    no_entropy: bool = False
    # random augmentation
    random_aug: float = 0
    aug_rmin: float = 400
    aug_rmax: float = 500
    aug_cmin: float = -2
    aug_cmax: float = 25
    cgap: float = 5
    rstd: float = 1
    cstd: float = 0.2


@dataclass
class OfflineModelCarCircleConfig(OfflineModelTrainConfig):
    pass


@dataclass
class OfflineModelAntRunConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineAntRun-v0"
    # augmentation param
    deg: int = 3
    max_reward: float = 1000.0
    max_rew_decrease: float = 150
    device: str = "cuda:2"


@dataclass
class OfflineModelDroneRunConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 200
    # training params
    task: str = "OfflineDroneRun-v0"
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((400.0, 10), (500.0, 20), (600.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 700.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelDroneCircleConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineDroneCircle-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 1000.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelCarRunConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarRun-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 600.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelAntCircleConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineAntCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelBallRunConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineBallRun-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1400.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelBallCircleConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineBallCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1000.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class OfflineModelCarButton1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class OfflineModelCarButton2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class OfflineModelCarCircle1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class OfflineModelCarCircle2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:0"


@dataclass
class OfflineModelCarGoal1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 50.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class OfflineModelCarGoal2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class OfflineModelCarPush1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class OfflineModelCarPush2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineCarPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class OfflineModelPointButton1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    state_encoder_path: str = "logs/OfflinePointButton1Gymnasium-v0-cost-10/sa_encoder-660a/sa_encoder-660a_state_AE"
    # state_encoder_path: str = "logs/OfflinePointButton1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-894b/sa_encoder_idm_loss_weight0.0-894b_state_AE"


@dataclass
class OfflineModelPointButton2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointButton2Gymnasium-v0-cost-10/sa_encoder-e7e4/sa_encoder-e7e4_state_AE"
    state_encoder_path: str = "logs/OfflinePointButton2Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-b874/sa_encoder_idm_loss_weight0.0-b874_state_AE"


@dataclass
class OfflineModelPointCircle1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 65.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointCircle1Gymnasium-v0-cost-10/sa_encoder-2db5/sa_encoder-2db5_state_AE"
    state_encoder_path: str = "logs/OfflinePointCircle1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-2c5d/sa_encoder_idm_loss_weight0.0-2c5d_state_AE"


@dataclass
class OfflineModelPointCircle2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 55.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"
    # state_encoder_path: str = "logs/OfflinePointCircle2Gymnasium-v0-cost-10/sa_encoder-1dbc/sa_encoder-1dbc_state_AE"
    state_encoder_path: str = "logs/OfflinePointCircle2Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-f4f4/sa_encoder_idm_loss_weight0.0-f4f4_state_AE"


@dataclass
class OfflineModelPointGoal1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:5"
    # state_encoder_path: str = "logs/OfflinePointGoal1Gymnasium-v0-cost-10/sa_encoder-82ac/sa_encoder-82ac_state_AE"
    state_encoder_path: str = "logs/OfflinePointGoal1Gymnasium-v0-cost-10/sa_encoder_idm_loss_weight0.0-62a4/sa_encoder_idm_loss_weight0.0-62a4_state_AE"


@dataclass
class OfflineModelPointGoal2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"


@dataclass
class OfflineModelPointPush1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelPointPush2Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflinePointPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelAntVelocityConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:1"
    need_rescale: bool = True

@dataclass
class OfflineModelAntVelocityV0Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:1"
    need_rescale: bool = True

@dataclass
class OfflineModelAntCircleV1Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineAntCircleGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 1200.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:3"
    need_rescale: bool = True


@dataclass
class OfflineModelHalfCheetahVelocityConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class OfflineModelHalfCheetahVelocityV0Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class OfflineModelHopperVelocityConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class OfflineModelHopperVelocityV0Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class OfflineModelSwimmerVelocityConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class OfflineModelSwimmerVelocityV0Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class OfflineModelWalker2dVelocityConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class OfflineModelWalker2dVelocityV0Config(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True

@dataclass
class OfflineModelPointRobotConfig(OfflineModelTrainConfig):
    # training params
    task: str = "PointRobot"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:4"
    need_rescale: bool = True


@dataclass
class OfflineModelEasySparseConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineMetadrive-easysparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelEasyMeanConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easymean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelEasyDenseConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easydense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelMediumSparseConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediumsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class OfflineModelMediumMeanConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediummean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelMediumDenseConfig(OfflineModelTrainConfig):
    # training params
    task: str = "OfflineMetadrive-mediumdense-v0"
    episode_len: int = 1000
    update_steps: int = 200_000


@dataclass
class OfflineModelHardSparseConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelHardMeanConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardmean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class OfflineModelHardDenseConfig(OfflineModelTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-harddense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


OfflineModel_DEFAULT_CONFIG = {
    # bullet_safety_gym
    "OfflineCarCircle-v0": OfflineModelCarCircleConfig,
    "OfflineAntRun-v0": OfflineModelAntRunConfig,
    "OfflineDroneRun-v0": OfflineModelDroneRunConfig,
    "OfflineDroneCircle-v0": OfflineModelDroneCircleConfig,
    "OfflineCarRun-v0": OfflineModelCarRunConfig,
    "OfflineAntCircle-v0": OfflineModelAntCircleConfig,
    "OfflineBallCircle-v0": OfflineModelBallCircleConfig,
    "OfflineBallRun-v0": OfflineModelBallRunConfig,
    # safety_gymnasium
    "OfflineCarButton1Gymnasium-v0": OfflineModelCarButton1Config,
    "OfflineCarButton2Gymnasium-v0": OfflineModelCarButton2Config,
    "OfflineCarCircle1Gymnasium-v0": OfflineModelCarCircle1Config,
    "OfflineCarCircle2Gymnasium-v0": OfflineModelCarCircle2Config,
    "OfflineCarGoal1Gymnasium-v0": OfflineModelCarGoal1Config,
    "OfflineCarGoal2Gymnasium-v0": OfflineModelCarGoal2Config,
    "OfflineCarPush1Gymnasium-v0": OfflineModelCarPush1Config,
    "OfflineCarPush2Gymnasium-v0": OfflineModelCarPush2Config,
    # safety_gymnasium: point
    "OfflinePointButton1Gymnasium-v0": OfflineModelPointButton1Config,
    "OfflinePointButton2Gymnasium-v0": OfflineModelPointButton2Config,
    "OfflinePointCircle1Gymnasium-v0": OfflineModelPointCircle1Config,
    "OfflinePointCircle2Gymnasium-v0": OfflineModelPointCircle2Config,
    "OfflinePointGoal1Gymnasium-v0": OfflineModelPointGoal1Config,
    "OfflinePointGoal2Gymnasium-v0": OfflineModelPointGoal2Config,
    "OfflinePointPush1Gymnasium-v0": OfflineModelPointPush1Config,
    "OfflinePointPush2Gymnasium-v0": OfflineModelPointPush2Config,
    # safety_gymnasium: velocity
    "OfflineAntVelocityGymnasium-v1": OfflineModelAntVelocityConfig,
    "OfflineAntCircleGymnasium-v1": OfflineModelAntCircleConfig,
    "OfflineHalfCheetahVelocityGymnasium-v1": OfflineModelHalfCheetahVelocityConfig,
    "OfflineHopperVelocityGymnasium-v1": OfflineModelHopperVelocityConfig,
    "OfflineSwimmerVelocityGymnasium-v1": OfflineModelSwimmerVelocityConfig,
    "OfflineWalker2dVelocityGymnasium-v1": OfflineModelWalker2dVelocityConfig,
    "OfflineAntVelocityGymnasium-v0": OfflineModelAntVelocityV0Config,
    "OfflineHalfCheetahVelocityGymnasium-v0": OfflineModelHalfCheetahVelocityV0Config,
    "OfflineHopperVelocityGymnasium-v0": OfflineModelHopperVelocityV0Config,
    "OfflineSwimmerVelocityGymnasium-v0": OfflineModelSwimmerVelocityV0Config,
    "OfflineWalker2dVelocityGymnasium-v0": OfflineModelWalker2dVelocityV0Config,
    "PointRobot": OfflineModelPointRobotConfig,
    # safe_metadrive
    "OfflineMetadrive-easysparse-v0": OfflineModelEasySparseConfig,
    "OfflineMetadrive-easymean-v0": OfflineModelEasyMeanConfig,
    "OfflineMetadrive-easydense-v0": OfflineModelEasyDenseConfig,
    "OfflineMetadrive-mediumsparse-v0": OfflineModelMediumSparseConfig,
    "OfflineMetadrive-mediummean-v0": OfflineModelMediumMeanConfig,
    "OfflineMetadrive-mediumdense-v0": OfflineModelMediumDenseConfig,
    "OfflineMetadrive-hardsparse-v0": OfflineModelHardSparseConfig,
    "OfflineMetadrive-hardmean-v0": OfflineModelHardMeanConfig,
    "OfflineMetadrive-harddense-v0": OfflineModelHardDenseConfig
}
